// content_script.js
(function() {
  // Practical email regex (may produce false positives)
  const emailRegex = /[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/g;

  // Extract visible text and some common attributes
  function getPageText() {
    let bodyText = "";
    try { bodyText = document.body ? document.body.innerText : ""; } catch(e){ bodyText = ""; }
    let attrsText = "";
    try {
      document.querySelectorAll("*").forEach(el => {
        if (el.getAttribute) {
          ["href", "src", "alt", "title", "data-mail"].forEach(attr => {
            const v = el.getAttribute(attr);
            if (v) attrsText += " " + v;
          });
        }
      });
    } catch(e) { attrsText = ""; }
    return (bodyText + " " + attrsText);
  }

  function extractEmails() {
    const text = getPageText();
    const matches = text.match(emailRegex) || [];
    const normalized = matches.map(m => m.trim().toLowerCase());
    const uniq = Array.from(new Set(normalized));
    return uniq;
  }

  // Save found emails in chrome.storage.local under 'foundEmails'
  function storeEmails(found) {
    if (!found || found.length === 0) return;
    chrome.storage.local.get(["foundEmails"], (data) => {
      const existing = Array.isArray(data.foundEmails) ? data.foundEmails : [];
      const combined = Array.from(new Set([...existing, ...found]));
      chrome.storage.local.set({ foundEmails: combined }, () => {
        try { console.log("[Email Extractor] Stored emails:", combined); } catch(e){}
      });
    });
  }

  // Run extraction immediately
  try {
    const emails = extractEmails();
    if (emails.length > 0) storeEmails(emails);
  } catch(err) {
    try { console.warn("[Email Extractor] extraction error", err); } catch(e){}
  }

  // Observe DOM changes for a short while to catch dynamically loaded content
  try {
    const observer = new MutationObserver((mutations) => {
      try {
        const newEmails = extractEmails();
        if (newEmails.length > 0) storeEmails(newEmails);
      } catch(e){}
    });
    observer.observe(document.documentElement || document.body, { childList: true, subtree: true });
    // Stop observing after 30 seconds (reduces overhead)
    setTimeout(() => observer.disconnect(), 30000);
  } catch(e){}
})();
